import os
import json
from pathlib import Path

import flet as ft
from dotenv import load_dotenv

from content.functions import load_decrypted_credentials, get_or_generate_key
from iLibrary import Library

#Information about Library: <NAME>
class Info(ft.Column):
    def __init__(self, page: ft.Page, library:str, content_manager):
        super().__init__(
            scroll=ft.ScrollMode.ADAPTIVE,
            horizontal_alignment=ft.CrossAxisAlignment.CENTER,


        )
        self.DOWNLOAD_PATH = Path.home() / "Downloads"
        self.current_page = page
        self.content_manager = content_manager
        self.library = library
        self.env_file_path = Path(__file__).parent / ".env"
        self.ENCRYPTION_KEY_STR = None
        self.db_credentials = None
        self.DB_DRIVER = None
        self.DB_USER = None
        self.DB_PASSWORD = None
        self.DB_SYSTEM = None
        self.DB_PORT = None

        self.list_container = ft.Column()
        self.input_card = self.list_container
        self.current_page.run_task(self._create_app_bar)
        self.progress_bar = ft.ProgressRing()
        self.progress_bar_container = ft.Container(self.progress_bar, alignment=ft.Alignment.CENTER)
        self.controls.append(self.progress_bar_container)

        # Start initialization
        self.current_page.run_task(self.async_init)

    def __exit__(self, exc_type, exc_val, exc_tb):

        self.progress_bar.visible = True
        self.progress_bar_container.visible = True
        self.current_page.update()
        self.lib.conn.close()

    async def _create_app_bar(self):
        server = await ft.SharedPreferences().get(key='server')
        self.current_page.appbar = ft.AppBar(
            title=ft.Text(f"Library Info: {self.library}"),
            leading=ft.IconButton(ft.Icons.ARROW_BACK, on_click=lambda e: self.current_page.run_task(self._go_back) ),
            actions = [
                # Reference the instance variable here
                ft.Text(f"Server: {server}"),
                ft.Container(width=60),
            ]
        )
        self.current_page.update()

    async def _go_back(self):
        try:
            from content.all_libraries import AllLibraries
            await self.content_manager(AllLibraries(
                self.current_page,
                content_manager=self.content_manager))
        except Exception as e:
            self.current_page.show_dialog(ft.SnackBar(
                content=ft.Text(
                    value=f"Failed to load library info: {e}",
                    color=ft.Colors.WHITE),
                bgcolor=ft.Colors.RED_ACCENT_400)
            )
    async def async_init(self):
        self.ENCRYPTION_KEY_STR = get_or_generate_key(self.env_file_path)
        load_dotenv(self.env_file_path, override=True)

        if os.getenv("ENCRYPTED_DB_CREDENTIALS"):
            self.db_credentials = load_decrypted_credentials(
                key=self.ENCRYPTION_KEY_STR,
                env_file_path=self.env_file_path,
            )

            if self.db_credentials:
                self.DB_DRIVER = self.db_credentials["driver"]
                self.DB_USER = self.db_credentials["user"]
                self.DB_PASSWORD = self.db_credentials["password"]
                self.DB_SYSTEM = self.db_credentials["system"]
                self.DB_PORT = self.db_credentials["port"]
                self.input_card.visible = True
                self.list_container.visible = True

                if self.input_card not in self.controls:
                    self.controls.extend([self.input_card])
                await self._get_info_about_library()
            else:
                pass
                #await self._show_setup_modal()
        else:
            pass
            #await self._show_setup_modal()

        self.update()

    async def _get_info_about_library(self):
        try:
            # 1. Clear previous results and show progress
            self.input_card.controls.clear()
            self.progress_bar.visible = True
            self.progress_bar_container.visible = True
            self.update()

            with Library(self.DB_USER, self.DB_PASSWORD, self.DB_SYSTEM, self.DB_DRIVER) as self.lib:
                # Fetch data
                result = self.lib.getLibraryInfo(library=self.library, wantJson=True)
                qFiles = self.lib.getFileInfo(library=self.library, qFiles=False)

                # --- ROBUST DATA PARSING ---
                # This handles the "Expecting value" error by checking types before loading
                def safe_parse(input_data):
                    if input_data is None:
                        return []
                    if isinstance(input_data, (list, dict)):
                        return input_data
                    if isinstance(input_data, str) and input_data.strip() == "":
                        return []
                    try:
                        return json.loads(input_data)
                    except Exception as e:
                        # If it's not JSON, return it as a list with an error message
                        return [{"error": f"Parse Error: {str(e)}", "raw": str(input_data)}]

                data = safe_parse(qFiles)
                library_info_data = safe_parse(result)

                # --- UI CONSTRUCTION ---
                result_text = ft.DataTable(
                    columns=[ft.DataColumn(label=ft.Text()),
                            ft.DataColumn(label=ft.Text())],
                    rows=[],
                    width=1000,
                    )
                panel_list = ft.ExpansionPanelList(
                    expand_icon_color=ft.Colors.PRIMARY,
                    elevation=0,
                    divider_color=ft.Colors.PRIMARY,
                    controls=[],
                )

                # Process File Info
                for item in data:
                    if "error" in item:
                        self.current_page.show_dialog(ft.SnackBar(
                            content=ft.Text(f"Notice: {item['error']}", color=ft.Colors.WHITE),
                            bgcolor=ft.Colors.RED_ACCENT_400
                        ))
                        # If it's a real error, show the raw output for debugging in the build
                        if "raw" in item:
                            panel_list.controls.append(ft.ExpansionPanel(
                                header=ft.ListTile(title=ft.Text("Raw Debug Data")),
                                content=ft.Container(content=ft.Text(item["raw"]))
                            ))
                        continue

                    content_column = ft.DataTable(
                    columns=[ft.DataColumn(label=ft.Text()),
                            ft.DataColumn(label=ft.Text())],
                    rows=[],
                    width=1000,
                    )
                    for key, value in item.items():
                        if not value or value == "None" or key == "OBJNAME":
                            continue
                        content_column.rows.append(
                            ft.DataRow(
                                cells=[
                                    ft.DataCell(ft.Text(f"{key.replace('_', ' ').title()}: ", weight=ft.FontWeight.BOLD, size=15)),
                                    ft.DataCell(ft.Text(value=str(value))),
                                ],
                            ),
                        )

                    panel_list.controls.append(
                        ft.ExpansionPanel(
                            header=ft.ListTile(
                                title=ft.Text(item.get("OBJNAME", "Unknown"), weight=ft.FontWeight.BOLD)),
                            can_tap_header=True,
                            bgcolor=ft.Colors.TRANSPARENT,
                            content=ft.Container(content=content_column, padding=10)
                        )
                    )

                # Process Library Info
                # Ensure library_info_data is a dict (if it was a list, take the first item)
                info_dict = library_info_data[0] if isinstance(library_info_data, list) else library_info_data

                for key, value in info_dict.items():
                    if key == "LIBRARY_SIZE" and value:
                        try:
                            mb = float(value) / 1000000
                            value = f"{round(mb, 2)} Mb"
                        except:
                            pass

                    if value is not None:
                        result_text.rows.append(
                            ft.DataRow(
                                cells=[
                                    ft.DataCell(ft.Text(f"{key.replace('_', ' ').title()}: ", weight=ft.FontWeight.BOLD,
                                                        size=15)),
                                    ft.DataCell(ft.Text(value=str(value))),
                                ],
                            ),
                        )

                # --- ICON AND LAYOUT SECTION ---
                img_icon = ft.Container(
                    # Arranges icon and layout elements in a stack
                    content=ft.Stack(
                        controls=[
                            ft.Container(
                                padding=ft.Padding.only(top=75),
                                content=ft.Card(
                                    elevation=10,
                                    content=ft.Container(
                                        padding=ft.Padding.all(25),
                                        content=ft.Column(
                                            horizontal_alignment=ft.CrossAxisAlignment.CENTER,
                                            controls=[
                                                ft.Container(height=40),
                                                ft.Text(f"Library {self.library.upper()}", size=20,
                                                        weight=ft.FontWeight.BOLD),
                                                ft.Text(f"Viewing details for {self.library}",
                                                        text_align=ft.TextAlign.CENTER),
                                                ft.Container(height=10),
                                                result_text,
                                                ft.Container(height=10),
                                                ft.Text("Files", size=18, weight=ft.FontWeight.BOLD,
                                                        style=ft.TextStyle(decoration=ft.TextDecoration.UNDERLINE)),
                                                panel_list
                                            ],
                                        ),
                                    ),
                                ),
                            ),
                            ft.Row(
                                controls=[
                                    ft.Container(
                                        padding=ft.Padding.only(top=30),
                                        content=ft.IconButton(
                                            icon=ft.Icons.DOWNLOAD,
                                            icon_color=ft.Colors.TRANSPARENT,
                                        ),
                                    ),
                                    ft.Container(
                                        width=130, height=130,
                                        bgcolor="#00ffe5",
                                        shape=ft.BoxShape.CIRCLE,
                                        alignment=ft.Alignment.CENTER,
                                        shadow=ft.BoxShadow(blur_radius=8, color="#00ffe5"),
                                        content=ft.Text(self.library[0:2].upper(), color="black",
                                                        weight=ft.FontWeight.BOLD, size=40),
                                    ),
                                    ft.Container(
                                        padding=ft.Padding.only(top=30),
                                        content=ft.IconButton(
                                            bgcolor="#00ffe5", icon_color="black", icon=ft.Icons.DOWNLOAD,
                                            on_click=lambda e: self.current_page.run_task(self._get_single_savefile,
                                                                                  self.library)
                                        ),
                                    )
                                ],
                                alignment=ft.MainAxisAlignment.SPACE_AROUND,
                            ),
                        ]
                    )
                )

                header_section = ft.Container(
                    content=ft.Column(horizontal_alignment=ft.CrossAxisAlignment.CENTER, controls=[img_icon]),
                    alignment=ft.Alignment.TOP_CENTER,
                    padding=ft.Padding.only(top=40),
                )

                self.input_card.controls.append(header_section)

        except Exception as e:
            self.input_card.controls.clear()
            # Display the full error for debugging
            self.input_card.controls.append(
                ft.Container(
                    padding=20,
                    content=ft.Column([
                        ft.Text("Connection or Data Error", weight=ft.FontWeight.BOLD, color="red"),
                        ft.Text(f"Details: {str(e)}", size=12)
                    ])
                )
            )

        finally:
            self.progress_bar.visible = False
            self.progress_bar_container.visible = False
            self.update()

    async def _get_single_savefile(self, name: str):
        """Get the Single Savefile of a Library """

        def download_save_file(library_name: str, savefile_name: str, description: str, version: str, authority: str,
                               download_path: str):
            try:
                self.current_page.pop_dialog()

                with Library(self.DB_USER, self.DB_PASSWORD, self.DB_SYSTEM, self.DB_DRIVER) as lib:
                    try:
                        lib.saveLibrary(
                            library=library_name,
                            saveFileName=savefile_name,
                            description=description,
                            localPath=download_path,
                            remPath=f'/home/{self.DB_USER.upper()}/',
                            authority=authority,
                            version=version,
                            remSavf=True,
                            getZip=True,
                            port=self.DB_PORT
                        )
                        self.current_page.run_task(ft.SharedPreferences().set, 'download_path', download_path)
                    except Exception as e:
                        self.current_page.show_dialog(ft.SnackBar(
                            content=ft.Text(f"Failed: {e}", color=ft.Colors.WHITE),
                            bgcolor=ft.Colors.RED_ACCENT_400))
                        return

                    self.current_page.show_dialog(ft.SnackBar(
                        content=ft.Text(f"Success, saved to: {download_path}", color=ft.Colors.WHITE),
                        bgcolor=ft.Colors.GREEN_ACCENT_400))

            except Exception as e:
                if hasattr(self, "input_card"):
                    self.input_card.controls.clear()
                    self.input_card.controls.append(ft.Text(f"Connection Error: {e}"))
                    self.current_page.update()

        # Ref fields for the download modal
        save_file_description_text_field_ref = ft.Ref[ft.TextField]()
        save_file_version_text_field_ref = ft.Ref[ft.TextField]()
        save_file_authority_text_field_ref = ft.Ref[ft.TextField]()
        save_file_download_path_text_field_ref = ft.Ref[ft.TextField]()
        save_file_name_text_field_ref = ft.Ref[ft.TextField]()
        save_file_download_path_text_field_ref.value = self.DOWNLOAD_PATH.as_posix()

        async def handle_get_directory_path(e: ft.Event[ft.Button]):
            download_path_field.value = await ft.FilePicker().get_directory_path()
            print(f"Selected directory: {save_file_download_path_text_field_ref.value}")
            download_path_field.update()


        download_path_field =ft.TextField(
                            ref=save_file_download_path_text_field_ref,
                            value=save_file_download_path_text_field_ref.value,
                            label="Download Path",
                            border_color=ft.Colors.PRIMARY,
                            on_click=lambda e: self.current_page.run_task(handle_get_directory_path, e),
                        )

        # text fields for the download modal

        self.download_modal = ft.AlertDialog(
            modal=True,
            title=ft.Text(f"Create Savefile from Library: {name}"),
            content=ft.Column([
                ft.TextField(
                    ref=save_file_name_text_field_ref,
                    label="Savefile name",
                    value=name,
                    border_color=ft.Colors.PRIMARY,
                ),
                ft.Container(height=5),
                ft.TextField(
                    ref=save_file_description_text_field_ref,
                    label="Description",
                    value="Saved by iLibrary",
                    border_color=ft.Colors.PRIMARY,
                ),
                ft.Container(height=5),
                ft.TextField(
                    ref=save_file_version_text_field_ref,
                    label="Version",
                    value="*CURRENT",
                    border_color=ft.Colors.PRIMARY,
                    helper="V7R1M0, V7R2M0, V7R3M0, V7R4M0, V7R5M0, V7R6M0 ..."
                ),
                ft.Container(height=5),
                ft.TextField(
                    ref=save_file_authority_text_field_ref,
                    label="Authority",
                    border_color=ft.Colors.PRIMARY,
                    value="*ALL",
                    helper="*EXCLUDE, *ALL, *CHANGE, *LIBCRTAUT, *USE"
                ),
                ft.Container(height=5),
                download_path_field,


            ],
                expand=True
            ),
            actions=[
                ft.TextButton("Close", on_click=lambda e: self.current_page.pop_dialog()),
                ft.TextButton(
                    content="Download",
                    style=ft.ButtonStyle(
                        bgcolor=ft.Colors.PRIMARY,
                        color=ft.Colors.ON_PRIMARY),
                    on_click=lambda e: download_save_file(
                        library_name=name,
                        savefile_name=save_file_name_text_field_ref.current.value,
                        description=save_file_description_text_field_ref.current.value,
                        version=save_file_version_text_field_ref.current.value,
                        authority=save_file_authority_text_field_ref.current.value,
                        download_path=save_file_download_path_text_field_ref.current.value
                    ), )
            ],
            actions_alignment=ft.MainAxisAlignment.END,
            on_dismiss=lambda e: print("Modal dialog dismissed!"),
        )
        self.current_page.show_dialog(self.download_modal)